program TAYLORSERIES;
{--------------------------------------------------------------------}
{  Alg4'1.pas   Pascal program for implementing Algorithm 4.1        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 4.1 (Evaluation of a Taylor Series).                    }
{  Section 4.1, Taylor Series and Calculation of Functions, Page 203 }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    GNmax = 130;
    Max = 100;
    MaxV = 100;
    Epsilon = 1E-11;

  type
    VECTOR = array[0..MaxV] of real;
    RVECTOR = array[0..GNmax] of real;
    LETTERS = string[200];
    STATUS = (Computing, Done, Working);

  var
    GNpts, Inum, K, Meth, N, Sub: integer;
    Close, Er2, Err, Nsum, Psum, Sum, Tol, X, X0: real;
    Rnum: real;
    Ans: CHAR;
    A, D: VECTOR;
    Xg, Yg, Yf: RVECTOR;
    Mess: LETTERS;
    Stat, State: STATUS;

  procedure TAYLOR (X0: real; D: VECTOR; X, Tol: real;
                    var Close, Sum: real; var Nsum, Psum: real;
                    var K: integer; N: integer);
    var
      Prod, Term: real;
  begin
    Close := 1;
    K := 0;
    Psum := 0;
    Nsum := 0;
    Sum := D[0];
    if 0 <= D[0] then
      Psum := Psum + D[0]
    else
      Nsum := Nsum + D[0];
    Prod := 1;
    if X = X0 then
      Close := 0;
    while (Close > Tol) and (K < N) do
      begin
        K := K + 1;
        Prod := Prod * (X - X0) / K;
        while (D[K] = 0) and (K < N) do
          begin
            K := K + 1;
            Prod := Prod * (X - X0) / K;
          end;
        Term := D[K] * Prod;
        if Term <> 0 then
          Close := ABS(Term);
        Sum := Sum + Term;
        if 0 <= Term then
          Psum := Psum + Term
        else
          Nsum := Nsum + Term;
      end;
  end;

  procedure POWER (X0: real; A: VECTOR; X, Tol: real; var Close, Sum: real; var Nsum, Psum: real; var K: integer; N: integer);
    var
      Prod, Term: real;
  begin
    Close := 1;
    K := 0;
    Psum := 0;
    Nsum := 0;
    Sum := A[0];
    if 0 <= A[0] then
      Psum := Psum + A[0]
    else
      Nsum := Nsum + A[0];
    Prod := 1;
    if X = X0 then
      Close := 0;
    while (Close > Tol) and (K < N) do
      begin
        K := K + 1;
        Prod := Prod * (X - X0);
        while (A[K] = 0) and (K < N) do
          begin
            K := K + 1;
            Prod := Prod * (X - X0);
          end;
        Term := A[K] * Prod;
        if Term <> 0 then
          Close := ABS(Term);
        Sum := Sum + Term;
        if 0 <= Term then
          Psum := Psum + Term
        else
          Nsum := Nsum + Term;
      end;
  end;

  function F (X: real): real;
  begin
    case Meth of
      1: 
        F := EXP(X);
      2: 
        F := COS(X);
      3: 
        F := SIN(X);
      4: 
        F := LN(X);
      5: 
        F := 0;
      6: 
        F := EXP(X);
      7: 
        F := COS(X);
      8: 
        F := SIN(X);
      9:
        F := 0;
    end;
  end;

  procedure MESSAGE (var Tol: real; var N: integer);
  begin
    CLRSCR;
    WRITELN('                    SERIES APPROXIMATIONS OF FUNCTIONS');
    WRITELN;
    WRITELN('         Evaluation of the Taylor series for the function  f(x)');
    WRITELN;
    WRITELN('     centered at  x .      The N-th partial sum is computed:');
    WRITELN('                   0 ');
    WRITELN('                                    2              k              N ');
    WRITELN('     P (x) = a + a (x-x ) + a (x-x ) +...+ a (x-x ) +...+ a (x-x )  ');
    WRITELN('      N       0   1    0     2    0         k    0         N    0   ');
    WRITELN('                                       (k)       ');
    WRITELN('     where the coefficients are  a  = f  (x )/k!   for  k=0,1,...,N');
    WRITELN('                                  K        0     ');
    WRITELN;
    WRITELN;
    WRITELN('          Convergence will be declared when consecutive partial');
    WRITELN;
    WRITELN('     sums differ by less than the preassigned value  TOL .');
    WRITELN;
    WRITELN('     that is,    | P (x) - P   (x) | < TOL');
    WRITELN('                    N       N-1  ');
    WRITELN;
    Tol := 0.000000001;
    Mess := '     ENTER the convergence criterion   TOL = ';
    WRITE(Mess);
    READ(Tol);
    if Tol < 0.000000001 then
      Tol := 0.000000001;
    WRITELN;
    Mess := '     ENTER the maximum number of terms   N = ';
    N := 50;
    WRITE(Mess);
    READ(N);
    if N < 1 then
      N := 1;
    if N > Max then
      N := Max;
    WRITELN;
  end;

  procedure GETDERIVATIVES (N: integer; var Meth: integer; var X0: real; var A, D: VECTOR);
    var
      K, MN: integer;
      D0, D1, P: real;
  begin
    Meth := 0;
    while Meth = 0 do
      begin
        CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN('     The Taylor series will be computed for:');
        WRITELN;
        WRITELN;
        WRITELN('     < 1 >  f(x) = exp(x)  expanded about  x0');
        WRITELN;
        WRITELN('     < 2 >  f(x) = cos(x)  expanded about  x0');
        WRITELN;
        WRITELN('     < 3 >  f(x) = sin(x)  expanded about  x0');
        WRITELN;
        WRITELN('     < 4 >  f(x) = Ln(x)   expanded about  x0 > 0');
        WRITELN;
        WRITELN('     < 5 >  ENTER the derivatives of f(x)');
        WRITELN;
        WRITELN('     < 6 >  Menu of Maclaurin series.');
        WRITELN;
        WRITELN;
        Mess := '        SELECT < 1 - 6 > ?  ';
        Meth := 1;
        WRITE(Mess);
        READ(Meth);
        if Meth < 1 then
          Meth := 1;
        if Meth > 6 then
          Meth := 6;
        if Meth = 6 then
          begin
            CLRSCR;
            WRITELN;
            WRITELN;
            WRITELN('     The Maclaurin series will be computed for:');
            WRITELN;
            WRITELN;
            WRITELN('      < 6 >  f(x) = exp(x)');
            WRITELN;
            WRITELN('      < 7 >  f(x) = cos(x)');
            WRITELN;
            WRITELN('      < 8 >  f(x) = sin(x)');
            WRITELN;
            WRITELN('     <  9 >  ENTER the derivatives of f(x)');
            WRITELN;
            WRITELN('     < 10 >  Menu of Taylor series.');
            WRITELN;
            WRITELN;
            Mess := '        SELECT < 1 - 10 > ?  ';
            Meth := 1;
            WRITE(Mess);
            READ(Meth);
            if (Meth < 6) or (Meth = 10) then
              Meth := 0;
            if Meth > 10 then
              Meth := 10;
          end;
      end;
    WRITELN('  ');
    if Meth < 6 then
      WRITELN('     You chose to compute the Taylor series of:')
    else
      WRITELN('     You chose to compute the Maclaurin series of:');
    WRITELN('  ');
    case Meth of
      1:
        WRITELN('        f(x) = exp(x)  expanded about  x0');
      2: 
        WRITELN('        f(x) = cos(x)  expanded about  x0');
      3: 
        WRITELN('        f(x) = sin(x)  expanded about  x0');
      4: 
        WRITELN('        f(x) = Ln(x)   expanded about  x0 > 0');
      5: 
        WRITELN('        Your own function f(x) expanded about x0');
      6: 
        WRITELN('        f(x) = exp(x) expanded about x0 = 0');
      7: 
        WRITELN('        f(x) = cos(x) expanded about x0 = 0');
      8: 
        WRITELN('        f(x) = sin(x) expanded about x0 = 0');
      9: 
        WRITELN('        Your own function f(x) expanded about x0 = 0');
    end;
    WRITELN;
    WRITELN;
    X0 := 0;
    if Meth < 6 then
      begin
        Mess := '     ENTER the center  x0 = ';
        WRITE(Mess);
        READ(X0);
      end;
    WRITELN;
    case Meth of
      1, 6: 
        begin (*EXP(X)*)
          D0 := EXP(X0);
          for K := 0 to N do
            D[K] := D0;
        end;
      2, 7: 
        begin (*COS(X)*)
          D0 := COS(X0);
          D1 := SIN(X0);
          for K := 0 to TRUNC(N / 4) do
            begin
              D[4 * K] := D0;
              D[4 * K + 1] := -D1;
              D[4 * K + 2] := -D0;
              D[4 * K + 3] := D1;
            end;
        end;
      3, 8: 
        begin (*SIN(X)*)
          D0 := SIN(X0);
          D1 := COS(X0);
          for K := 0 to TRUNC(N / 4) do
            begin
              D[4 * K] := D0;
              D[4 * K + 1] := D1;
              D[4 * K + 2] := -D0;
              D[4 * K + 3] := -D1;
            end;
        end;
      4: 
        begin (*LN(X)*)
          if X0 <= 0 then
            X0 := 1;
          A[0] := LN(X0);
          A[1] := 1 / X0;
          P := X0;
          for K := 2 to N do
            begin
              P := -P * X0;
              A[K] := 1 / P / K;
            end;
        end;
      5, 9: 
        begin
          if X0 <> 0 then
            begin
              WRITELN('Enter the ', N + 1, ' derivatives for the Taylor series');
              WRITELN('of  f(x)  expanded about the center value  x0 = ', X0);
            end
          else
            begin
              WRITELN('Enter the ', N + 1, ' derivatives for the Maclaurin series');
              WRITELN('of  f(x)  expanded about the center value  x0 = ', X0);
            end;
          WRITELN;
          WRITELN('D(0) , D(1) , D(2) ,..., D(N)');
          WRITELN;
          for K := 0 to N do
            begin
              WRITE('D(', K, ') = ');
              READ(D[K]);
            end;
        end;
    end;
    if Meth <> 4 then
      begin
        A[0] := D[0];
        P := 1;
        MN := N;
        if N > 33 then
          MN := 33;
        for K := 1 to MN do
          begin
            P := P * K;
            A[K] := D[K] / P;
          end;
        for K := MN + 1 to N do
          A[K] := 0;
      end;
  end;

  procedure InputX (X0: real; var X: real);
  begin
    WRITELN;
    WRITELN;
    WRITELN('     Give the value  x  for which you want to compute  P (x)');
    WRITELN('                                                        N');
    WRITELN;
    Mess := '     ENTER  x = ';
    X := X0;
    WRITE(Mess);
    READ(X);
  end;

  procedure PRINTMAC (A: VECTOR; N: integer);
    var
      K, U, V: integer;
  begin
    WRITELN;
    WRITELN('     The Maclaurin polynomial of degree  N = ', N : 2, '  is:');
    WRITELN;
    case N of
      1: 
        begin
          WRITELN('P(x)  =  a  +  a x');
          WRITELN('          0     1');
        end;
      2: 
        begin
          WRITELN('                           2');
          WRITELN('P(x)  =  a   +  a x  +  a x');
          WRITELN('          0      1       2');
        end;
      3: 
        begin
          WRITELN('                           2        3');
          WRITELN('P(x)  =  a   +  a x  +  a x  +  a  x');
          WRITELN('          0      1       2       3');
        end;
      4, 5, 6, 7, 8, 9: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '        ', N : 1);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a x   +  a x');
          WRITELN('          0      1       2            ', N - 1 : 1, '        ', N : 1);
        end;
      10: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '        ', N : 1);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a x  +  a  x');
          WRITELN('          0      1       2            ', N - 1 : 1, '       ', N : 1);
        end;
      else
        begin
          WRITELN('                           2             ', N - 1 : 2, '        ', N : 2);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a  x   +  a  x');
          WRITELN('          0      1       2            ', N - 1 : 2, '        ', N : 2);
        end;
    end;
    WRITELN;
    for K := 0 to TRUNC(N / 2) do                {Print the coefficients}
      begin
        U := 2 * K;                                {of P(x) in two columns}
        V := 2 * K + 1;
        if U <= N then
          begin
            WRITE('A(', U : 2, ') =', A[U], '         ');
            if V <= N then
              WRITELN('A(', V : 2, ') =', A[V])
            else
              WRITELN;
          end;
      end;
    WRITELN;
    WRITELN;
  end;

  procedure PRINTTAY (A: VECTOR; X0: real; N: integer);
    var
      K, U, V: integer;
  begin
    WRITELN;
    WRITELN('     The Taylor polynomial of degree N = ', N : 2);
    WRITELN;
    WRITELN('     centered about the value  x  =', X0, '  is:');
    WRITELN('                                0');
    WRITELN;
    case N of
      1: 
        begin
          WRITELN('P(x)  =  a  +  a (x-x )');
          WRITELN('          0     1    0');
        end;
      2: 
        begin
          WRITELN('                                     2');
          WRITELN('P(x)  =  a   +  a (x-x )  +  a (x-x )');
          WRITELN('          0      1    0       2    0');
        end;
      3: 
        begin
          WRITELN('                                     2            3');
          WRITELN('P(x)  =  a   +  a (x-x )  +  a (x-x )  +  a (x-x )');
          WRITELN('          0      1    0       2    0       3    0');
        end;
      4, 5, 6, 7, 8, 9: 
        begin
          WRITELN('                                 2                ', N - 1 : 1, '           ', N : 1);
          WRITELN('P(x)  =  a  + a (x-x ) + a (x-x )  +...+  a (x-x )  + a (x-x )');
          WRITELN('          0    1    0     2    0           ', N - 1 : 1, '    0      ', N : 1, '    0');
        end;
      10:
        begin
          WRITELN('                                 2                ', N - 1 : 1, '            ', N : 1);
          WRITELN('P(x)  =  a  + a (x-x ) + a (x-x )  +...+  a (x-x )  + a  (x-x )');
          WRITELN('          0    1    0     2    0           ', N - 1 : 1, '    0      ', N : 1, '    0');
        end;
      else
        begin
          WRITELN('                                 2                 ', N - 1 : 2, '            ', N : 2);
          WRITELN('P(x)  =  a  + a (x-x ) + a (x-x )  +...+  a  (x-x )   + a  (x-x )');
          WRITELN('          0    1    0     2    0           ', N - 1 : 2, '    0       ', N : 2, '    0');
        end;
    end;
    WRITELN;
    for K := 0 to TRUNC(N / 2) do                {Print the coefficients}
      begin
        U := 2 * K;                                {of P(x) in two columns}
        V := 2 * K + 1;
        if U <= N then
          begin
            WRITE('A(', U : 2, ') =', A[U], '         ');
            if V <= N then
              WRITELN('A(', V : 2, ') =', A[V])
            else
              WRITELN;
          end;
      end;
    WRITELN;
    WRITELN;
  end;

  procedure RESULT (X0: real; D: VECTOR; X, Tol, Close: real; Sum, Nsum, Psum: real; K, N: integer);
  begin
    WRITELN;
    WRITELN;
    WRITE('          You  chose  to  approximate ');
    case Meth of
      1: 
        WRITELN(' exp(', X : 15 : 8, '  ).');
      2: 
        WRITELN(' cos(', X : 15 : 8, '  ).');
      3: 
        WRITELN(' sin(', X : 15 : 8, '  ).');
      4: 
        WRITELN(' Ln(', X : 15 : 8, '  ).');
      5: 
        WRITELN('your function  F(', X : 15 : 8, '  ).');
      6: 
        WRITELN(' exp(', X : 15 : 8, '  ).');
      7: 
        WRITELN(' cos(', X : 15 : 8, '  ).');
      8: 
        WRITELN(' sin(', X : 15 : 8, '  ).');
      9: 
        WRITELN('your function F(', X : 15 : 8, '  ).')
    end;
    Err := ABS(Psum) * Epsilon;
    Er2 := ABS(Nsum) * Epsilon;
    if Er2 > Err then
      Err := Er2;
    WRITELN;
    if X0 <> 0 then
      begin
        WRITELN('     Using a Taylor series with center   x  = ', X0 : 15 : 8);
        WRITELN('                                          0');
      end
    else
      begin
        WRITELN('     Using a Maclaurin series with center   x  = ', X0 : 15 : 8);
        WRITELN('                                             0');
      end;
    WRITELN;
    if (Close < Tol) and (K <= N) then
      begin
        WRITELN('     The value of the Taylor polynomial of degree ', K, ' IS:');
        WRITELN;
        WRITELN('                P   (', X : 15 : 8, '  )  = ', Sum : 15 : 8);
        WRITELN('          ', K);
        WRITELN;
        WRITELN('     Consecutive partial sums are closer than ', Close : 15 : 8);
        WRITELN;
        WRITELN('          Convergence has been achieved.');
        WRITELN('          Consecutive partial sums differ');
        WRITELN('          by less than the given value  TOL = ', Tol : 15 : 8);
        WRITELN;
        WRITELN('          Round off error for the sum is  +-  ', Err : 15 : 8);
      end
    else
      begin
        WRITELN('     The value of the Taylor polynomial of degree ', K, ' is:');
        WRITELN;
        WRITELN('            P  (', X : 15 : 8, '  )  = ', Sum : 15 : 8);
        WRITELN('               ', K);
        WRITELN;
        WRITELN('     Consecutive partial sums differ by  ', Close : 15 : 8);
        WRITELN;
        WRITELN('     Convergence has NOT been achieved.');
        WRITELN('     Consecutive partial sums differ');
        WRITELN('     By MORE than the given value  TOL = ', Tol : 15 : 8);
        WRITELN;
        WRITELN('     Round off error for the sum is  +-  ', Err : 15 : 8);
        WRITELN;
        WRITELN('     Perhaps you should choose a value of N larger than ', N : 3);
        WRITELN('     or you should use a center  x  closer to  x.');
        WRITELN('                                  0');
      end;
  end;

begin                                      {Start of main program}
  Stat := Working;
  while (Stat = Working) do
    begin
      MESSAGE(Tol, N);
      GETDERIVATIVES(N, Meth, X0, A, D);
      State := Computing;
      while (State = Computing) do
        begin
          InputX(X0, X);
          if (Meth <> 4) or (Meth <> 9) or (Meth <> 10) then
            TAYLOR(X0, D, X, Tol, Close, Sum, Nsum, Psum, K, N);
          if (Meth = 4) or (Meth = 9) or (Meth = 10) then
            POWER(X0, A, X, Tol, Close, Sum, Nsum, Psum, K, N);
          RESULT(X0, A, X, Tol, Close, Sum, Nsum, Psum, K, N);
          WRITE('     Want  to evaluate  the  series  again ?  <Y/N>  ');
          READLN(Ans);
          WRITELN(' ');
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
        end;
      WRITE('     Want to see the  series  coefficients ?  <Y/N>  ');
      READLN(Ans);
      if (Ans = 'Y') or (Ans = 'y') then
        begin
          WRITE('     You  chose  to  approximate ');
          case Meth of
            1: 
              WRITELN(' exp(x).');
            2: 
              WRITELN(' cos(x).');
            3: 
              WRITELN(' sin(x).');
            4: 
              WRITELN(' Ln(x)   where  x0 > 0');
            5: 
              WRITELN('your own Taylor series.');
            6: 
              WRITELN(' exp(x)');
            7: 
              WRITELN(' cos(x)');
            8: 
              WRITELN(' sin(x)');
            9: 
              WRITELN('your own Maclaurin series.');
          end;
          if X0 = 0 then
            PRINTMAC(A, N)
          else
            PRINTTAY(A, X0, N);
        end;
      WRITE('     Want  to  try  a  different  function ?  <Y/N>  ');
      READLN(Ans);
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done
    end;
end.                                         {End of main program}

